/*
 * MyString.cpp
 *
 * Implementation of the MyString class
 * Version 0.2
 *   --Add constructor which takes string
 *   --Add destructor
 * Version 0.4
 *   --Add various operator overloads
 *   --Add findChar
 * Version 0.5
 *   --Fixed for problems with various compilers
 * Version 0.6
 *   --Added operators from Assignment 6
 *   --Fixed up overloaded >> operator to deal with file input
 *   --Added getAllocatedSpace() and setAllocatedSpace()
 * Version 0.7
 *   --Fixed problem of not always catching EOF in overloaded >>
 *   --Added "return *this" statements at the end of overloaded +=
 *     operators in which they were missing
 *   --Fixed bug in which setValue() wasn't accounting for the NULL
 *     byte when determining if enough space existed.
 * Version 0.8
 *   --Added #ifdef _MYSTRING to header file
 */

// Uncomment the following line if you are using Visual C++ (Windows)
//#include "stdafx.h"


#include <iostream>
#include <string>
using namespace std;

#include "MyString.h"


// MEMORY MANAGEMENT SECTION
// MyStringFreeList class allows us to keep a list of 
// MyString instances which have been freed (so that they
// may be reused)  We define a cast to MyString * so that 
// the new operator my return an instance of this structure.
class MyStringFreeList
{
public:
  operator MyString *()
  {
    return &thisData;
  };
  
public:
  MyString thisData;
  MyStringFreeList *theNextString;
};


MyStringFreeList *nextFreeMyString = NULL;		// List of allocated space (array)
MyStringFreeList *lastMyString = NULL;		// Location of last new item we have room for
MyStringFreeList *deletedMyStrings = NULL;		// List of free strings ready for re-use

void *MyString::operator new(size_t size)
{
  // Make sure size requested matches a MyString size
  if (size != sizeof(MyString))
  {
    return ::operator new(size);
  }
  
  // First, check the free list
  if (deletedMyStrings)
  {
    void *freePtr = deletedMyStrings;
    deletedMyStrings = deletedMyStrings->theNextString;
    return freePtr;
  }
  
  // Second, check to see if there are any more "array" elements
  if (nextFreeMyString < lastMyString)
  {
    void *freePtr = nextFreeMyString++;
    return freePtr;
  }
  
  // Allocate another chunk
  char *newChunk = new char[10000 * sizeof(MyStringFreeList)];
  nextFreeMyString = (MyStringFreeList *) newChunk;
  lastMyString = nextFreeMyString + 10000;
  void *freePtr = nextFreeMyString++;
  return freePtr;
}


void MyString::operator delete(void *d)
{
  ((MyStringFreeList *)d)->theNextString = (MyStringFreeList *)deletedMyStrings;
  deletedMyStrings = (MyStringFreeList *)d;
}
// END MEMORY MANAGEMENT
// Standard constructor, makes our string a null-terminated,
// 0 length string
MyString::MyString()
{
  storagePtr = &initialStorage[0];
  allocatedSpace = 20;
  stringLength = 0;
  initialStorage[0] = '\0';
}

// Oddball constructor, takes an integer parameter to define the 
// initial storage size of MyString
MyString::MyString(int initSize)
{
  storagePtr = &initialStorage[0];
  allocatedSpace = 20;
  stringLength = 0;
  initialStorage[0] = '\0';
  growStorage(initSize);
}


// Constructor which allows an initial value to be set
MyString::MyString(string initialValue)
{
  storagePtr = &initialStorage[0];
  allocatedSpace = 20;
  stringLength = 0;
  initialStorage[0] = '\0';
  setValue(initialValue);
}

// Copy Constructor
MyString::MyString(MyString &aCopy)
{
  stringLength = aCopy.stringLength;
  if (aCopy.allocatedSpace > 20)
  {
    storagePtr = new char[aCopy.allocatedSpace];
    if (!storagePtr)
    {
      // Couldn't allocate space, return an empty string
      allocatedSpace = 20;
      storagePtr = &initialStorage[0];
      *storagePtr = '\0';
      stringLength = 0;
      return;
    }
    
    strcpy(storagePtr,aCopy.storagePtr);
    allocatedSpace = aCopy.allocatedSpace;
    return;
  }
  
  strcpy(initialStorage,aCopy.initialStorage);
  allocatedSpace = 20;
  storagePtr = &initialStorage[0];
}
    

// Standard destructor, frees dynamically allocated memory
MyString::~MyString()
{
  if (allocatedSpace > 20)
  {
//    cout << "MyString:  Freeing '" << MakeString() << "' " << endl;
//    cout << "MyString:  Freeing " << allocatedSpace << " bytes of memory! " << endl;
    delete storagePtr;
  }
}


// Overload =
MyString &MyString::operator=(const MyString &sourceStr)
{  
  // Make sure we actually have two pointers
  if (this != &sourceStr)
    setValue(sourceStr.MakeString());
  return *this; 
}

MyString &MyString::operator=(string aStr)
{
  setValue(aStr);
  return *this;
}

MyString &MyString::operator=(char *cstr)
{
  string tempstr = cstr;
  setValue(tempstr);
  return *this;
}


// Overload ~ to return string length
int MyString::operator~()
{
  return stringLength;
}
  
// Overload (string) cast to return a C++ string class instance
MyString::operator string() const
{
  return MakeString();
}


// overload (char *) cast
MyString::operator char *() const
{
  return storagePtr;
}
  
// Overload [] to give us array like functionality   
char &MyString::operator[](int i) const
{
  return element(i);
}


// MakeString returns a C++ style string constructed from
// our internal "C" style string
string MyString::MakeString() const
{
  string returnVal = storagePtr;
  return returnVal;
}

// MakeInt utilized the C Standard library function "atoi" 
// to convert our string to an integer 
int MyString::MakeInt() const
{
  return( atoi(storagePtr) );
}

// readString tells cin to read in a new line of text
// into our string
int MyString::readString()
{
  return readString(allocatedSpace-1);
}

int MyString::readString(int maxSize)
{
  // Make sure we have enough storage 
  if (!growStorage(maxSize))
    return 0;
    
  cin.getline(storagePtr,maxSize);
  stringLength = strlen(storagePtr);
  return stringLength;
}


//
// MyString::findChar is used to find the first index at which the
// specified character appears.
//
int MyString::findChar(char c,int startPos,int stopPos)
{
  int stopAt = stringLength -1;
  
  if ((stopPos > 0) && (stopPos < stopAt))
    stopAt = stopPos;
  
    
  // Search for the specified character
  for (int k=startPos; k<=stopAt; k++)
    if (storagePtr[k] == c)
      return k;  // Found it!  Return the index to the caller

  // Didnt find it, return -1;
  return -1;
}

//
// element lets us get at an individual character in our string
//
char &MyString::element(int i) const
{

  char badVal = (char)-1;
  if ((i > 0) && (i < stringLength))
    return( storagePtr[i] );
  return badVal;
}

// setValue would provide a way to assign an existing 
// value to our string (that is, without reading in from
// standard console)
bool MyString::setValue(string cppString)
{
  int spaceNeeded = cppString.length();
   
  // make sure we have enough space to hold the new string
  // Don't forget to leave room for the NULL byte
  if (spaceNeeded > (allocatedSpace-1))
  {
     if (!growStorage(spaceNeeded+1))
        return false;
  }
  
  strcpy(storagePtr,cppString.c_str());
  stringLength = spaceNeeded;
  return true;
}

// growStorage is a private member function which is used
// to grow the buffer used to store our string when needed.
// We grow the buffer in blocks of 32 bytes to minimize the 
// amount of memory allocations we need to do.
bool MyString::growStorage(int spaceNeeded)
{
  // Make sure it really needs to grow
  if (spaceNeeded < allocatedSpace)
    return true;
    
  // grow in chunks of 32 bytes
  int padding = spaceNeeded % 32;
  spaceNeeded += (32 - padding);
  
  // allocate new storage
  char *newStoragePtr = new char[spaceNeeded];
  
  // fail if new storage allocation fails without
  // touching the existing contents of our string
  if (!newStoragePtr)
    return false;
    
  // copy the contents of the old string into the new
  strcpy(newStoragePtr,storagePtr);
  
  // delete the old string only if it's pointing at 
  // dynamically allocated memory.
  if (allocatedSpace != 20)
    delete storagePtr;
  
  // Take note of the new size of our allocated space
  allocatedSpace = spaceNeeded;
  
  // store the pointer to the new storage in our class
  storagePtr = newStoragePtr;
  
  return true;
}

MyString &MyString::operator+=(const MyString &str)
{
  *this = *this + str;
  return *this;
}
  
MyString &MyString::operator+=(const string &cppStr)
{
  MyString tempStr(cppStr);
  *this = *this + tempStr; 

  return *this;
}

MyString &MyString::operator+=(const char *cStr)
{
  MyString tempStr(cStr);
  *this = *this + tempStr;

  return *this;
}


